#!/bin/sh

VERSION="3.0.4"
SOFTUSB="SoftUSB3"

log() {
	local RED="\033[0;31m"
	local NONE="\033[0m"
	local LIGHT_PURPLE="\033[1;35m"
	local BLUE="\033[0;34m"
	if [ $# -eq 2 ]; then
		case $1 in
		warning)"$BUSYBOX" printf "${LIGHT_PURPLE}### $2${NONE}\n";;
		error)	"$BUSYBOX" printf "${RED}!!! $2${NONE}\n";;
		info)	"$BUSYBOX" printf "${BLUE}--> ${NONE}$2\n";;
		*)	"$BUSYBOX" printf "${NONE}$2\n";;
		esac
	else
		"$BUSYBOX" printf "${NONE}$1\n"
	fi
}

softusb_usage(){
#syntax softusb_usage()
	echo "$0 <command>
	commands:
		install [128m] - install $SOFTUSB
		uninstall - uninstall $SOFTUSB
		update - update image information
		set reset <usb|ide|both>
		set label <label>
		set flash <auto|e2|pvrVER> [<dev>] - set flash image
		set env <name> <value> - set custom uboot entries
		set net <ip> <netmask> <gateway> [<server>] - set network configuration
		set mac <mac> - set mac address
		set default <num> - set default boot position
		set savelastboot <yes|no> - save last boot
		status - print current configuration

Version: $VERSION"
}

softusb_configinit(){
	rm -f /tmp/config.tmp
	if [ -f "$DIR/config.ub" ]; then
		dd if="$DIR/config.ub" of=/tmp/config.tmp bs=72 skip=1 >/dev/null 2>&1
	else
		rm -f /tmp/config.tmp 2>/dev/null
		touch /tmp/config.tmp
	fi
}

softusb_configset(){
	#remove old value
	local file="`cat /tmp/config.tmp`"
	echo "$file" | grep -v -i "setenv $1 " > /tmp/config.tmp
	#add new value
	if [ ! -z "$2" ]; then
		file="`cat /tmp/config.tmp`"
		"$BUSYBOX" printf "$file\nsetenv $1 '$2'" > /tmp/config.tmp
	fi
}

softusb_configsetclear(){
	#remove old value
	local file="`cat /tmp/config.tmp`"
	echo "$file" | grep -v -i "setenv $1 " > /tmp/config.tmp
	#add new value
	file="`cat /tmp/config.tmp`"
	"$BUSYBOX" printf "$file\nsetenv $1" > /tmp/config.tmp
}

softusb_configcommit(){
	#remove blank lines
	file="`cat /tmp/config.tmp`"
	echo "$file" | grep -v ^$ > /tmp/config.tmp
	"$DIR/files/mkimage" -A sh -T script -C none -n "$SOFTUSB Config" -d /tmp/config.tmp "$DIR/config.ub" >/dev/null
}

softusb_mount(){
#syntax softusb_mount(dev,dir)
	local DEV="`\"$BUSYBOX\" findfs $1`"
	mkdir -p "$2"
	[ -f /etc/mtab ] || ln -s /proc/mounts /etc/mtab
	if ! "$BUSYBOX" mount -r $DEV "$2" 2>/dev/null; then
		"$BUSYBOX" mount $DEV "$2" 2>/dev/null
	fi
	return $?
}

softusb_umount(){
#syntax softusb_umount(dev,dir)
	"$BUSYBOX" umount "$2" 2>/dev/null
	rmdir "$2" 2>/dev/null
	return $?
}

softusb_detect(){
	unset DETECT
	[ -f "$1/usr/local/bin/enigma2" ] && DETECT="e2_$BOXMEM"
	[ -f "$1/usr/local/bin/neutrino" ] && DETECT="neutrino_$BOXMEM"
	if [ -f "$1/app/program/checkbootparam" ]; then
		CRC32="`\"$BUSYBOX\" crc32 \"$1/app/program/checkbootparam\" | cut -d\  -f1`"
		case $CRC32 in
		504392027)	DETECT=pvr104;;
		82900272)	DETECT=pvr105;;
		3446710112)	DETECT=pvr106;;
		3205506864)	DETECT=pvr107;;
		esac
	fi
	if [ -f "$1/app/app/checkbootparam" ]; then
		CRC32="`\"$BUSYBOX\" crc32 \"$1/app/app/checkbootparam\" | cut -d\  -f1`"
		case $CRC32 in
		4284041417)	DETECT=pvr200;;
		177148495)	DETECT=pvr202;;
		esac
	fi
}

softusb_install(){
#syntax softusb_install(...)
	local config
	log info "Installing mini-uboot..."
	softusb_checkuboot
	CHECKUBOOT=$?
	if [ $CHECKUBOOT -eq 0 ]; then
		softusb_envuboot
		[ "$1" = "128m" ] && BOXMEM=128 || BOXMEM=64
		local UB_FILE="mini-uboot$BOXMEM.ub"
		if [ ! -f "$DIR/files/$UB_FILE" ]; then
			log error "Could not find files/$UB_FILE file !!!"
			exit 1
		fi
		log info "Backup of uboot configuration..."
		dd if="/dev/$UBOOT1CONF_MTDBLOCK" of=/tmp/ubootconf.bin bs=65536 count=1 2>/dev/null
		SUM="`\"$BUSYBOX\" crc32 /tmp/ubootconf.bin`"
		CRC32_DEC=`echo "$SUM" | cut -d\  -f1`
		CRC32="`\"$BUSYBOX\" printf %x $CRC32_DEC`"
		UB_BACKUP="$DIR/bootconf`date +%Y%m%d`.img"
		echo -n "MARU`date +%Y%m%d`btcf" > "$UB_BACKUP"
		cat /tmp/ubootconf.bin >> "$UB_BACKUP"
		"$BUSYBOX" printf "\x`echo $CRC32 | cut -b 7,8`" >> "$UB_BACKUP"
		"$BUSYBOX" printf "\x`echo $CRC32 | cut -b 5,6`" >> "$UB_BACKUP"
		"$BUSYBOX" printf "\x`echo $CRC32 | cut -b 3,4`" >> "$UB_BACKUP"
		"$BUSYBOX" printf "\x`echo $CRC32 | cut -b 1,2`" >> "$UB_BACKUP"

		log info "Flashing mini-uboot..."
		cat "$DIR/files/$UB_FILE" >> /tmp/ubootconf.bin
		if dd if=/tmp/ubootconf.bin of=/dev/$UBOOT1CONF_MTDBLOCK bs=65536 count=2 2>/dev/null; then
			log info "Mini-uboot flashed successfully"
		else
			log error "Mini-uboot flashing failed !"
			exit 1
		fi
		sync && sleep 1
		SUM="`\"$BUSYBOX\" crc32 \"$DIR/files/$UB_FILE\"`"
		CRC32_DEC=`echo "$SUM" | cut -d\  -f1`
		SIZE_DEC=`echo "$SUM" | cut -d\  -f2`
		CRC32="`\"$BUSYBOX\" printf %x $CRC32_DEC`"
		SIZE="`\"$BUSYBOX\" printf %x $SIZE_DEC`"

		if [ -z "$CRC32" ]; then
			log error "Error calculating crc !\n$CRC32_DEC\n$SIZE\n$CRC32"
			exit 1
		fi

		softusb_envuboot
		#backup original bootcmd
		local BOOTCMD="`\"$PRINTENV\" bootcmd_0 2>/dev/null | cut -d= -f2-`"
		local BOOTARGS="`\"$PRINTENV\" bootargs 2>/dev/null | cut -d= -f2-`"
		if [ -z "$BOOTCMD" ]; then
			"$SETENV" bootargs_0 "$BOOTARGS" >/dev/null 2>&1
			BOOTCMD="`\"$PRINTENV\" bootcmd 2>/dev/null | cut -d= -f2-`"
			"$SETENV" bootcmd_0 "run bootargs_0; $BOOTCMD" >/dev/null 2>&1
			"$SETENV" bootdesc_0 'Flash' >/dev/null 2>&1
			"$SETENV" bootdesc_default '0' >/dev/null 2>&1
		fi
		"$SETENV" bootcmd "crc32 A0030000 $SIZE A5000000; mw A5000004 $CRC32; if cmp A5000000 A5000004 1; then bootm A0030000; fi; run bootcmd_0; bootm A0040000" >/dev/null 2>&1
		"$SETENV" softusb '$fsload $fsinterface $fsdev A5300000 /softusb/softusb.ub' >/dev/null 2>&1
		"$SETENV" busreset 'ide reset; usb reset' >/dev/null 2>&1
		"$SETENV" label 'UFS910' >/dev/null 2>&1
		"$SETENV" minibootcmd 'run busreset; if findfs $label; then $fsload $fsinterface $fsdev a5000000 /softusb/config.ub; autoscr a5000000; fi; bootmenu;' >/dev/null 2>&1
		"$SETENV" bootdesc_savelast "1" >/dev/null 2>&1
		"$SETENV" fsload "fatload" >/dev/null 2>&1
		#timeout set to 1sec
		"$SETENV" bootdelay '1' >/dev/null 2>&1

		local ETHADDR="`ifconfig eth0 | grep HWaddr | cut -dW -f2- | cut -c 6-`"
		local IPADDR="`\"$PRINTENV\" ipaddr 2>/dev/null | cut -d= -f2-`"
		[ -z "$ETHADDR" ] || "$SETENV" ethaddr "$ETHADDR" >/dev/null 2>&1
		if [ -z "$IPADDR" ]; then
			log warning "Please run \"$0 set net\" and configure your network"
		fi


		log warning "Please run \"$0 set flash\" and configure your flash image"

		local file="`cat \"$DIR/softusb.conf\" 2>/dev/null`"
		local file="`echo \"$file\" | grep -v -i 'BOXMEM='`"
		"$BUSYBOX" printf "BOXMEM=$BOXMEM\n$file" > "$DIR/softusb.conf"
		sleep 1
		log info "$SOFTUSB configured"
	else
		log error "Problem with kernel/uboot"
		exit 1
	fi
}

softusb_uninstall(){
	log info "Uninstalling $SOFTUSB..."
	echo "/dev/$UBOOT1CONF_MTD 0x00000000 0x00010000 0x10000" > /tmp/su_env.config
	local BOOTCMD="`\"$PRINTENV\" bootcmd_0 2>/dev/null | cut -d= -f2-`"
	if [ -z "$BOOTCMD" ]; then
		"$SETENV" bootcmd 'bootm A0040000' >/dev/null 2>&1
	else
		"$SETENV" bootcmd "$BOOTCMD" >/dev/null 2>&1
		"$SETENV" bootcmd_0 >/dev/null 2>&1
	fi
	log info "$SOFTUSB uninstalled"
}

softusb_set() {
	if [ $CHECKUBOOT -eq 0 ]; then
		case $1 in
		reset)	softusb_setreset "$2"; return 0;;
		label)	softusb_setlabel "$2"; return 0;;
		flash)	softusb_setflash "$2" "$3"; return 0;;
		env)	softusb_setenv "$2" "$3"; return 0;;
		net)	softusb_setnet "$2" "$3" "$4" "$5"; return 0;;
		mac)	softusb_setmac "$2"; return 0;;
		default)	softusb_setdefault "$2"; return 0;;
		savelastboot)	softusb_setsavelastboot "$2"; return 0;;
		esac
	else
		log error "Problem with accessing uboot"
		exit 1
	fi

	log error "Unknown command $1"
	exit 1
}

softusb_setlabel() {
	local LABEL
	if [ -z $1 ]; then
		LABEL=UFS910
	else
		LABEL=$1
	fi

	"$SETENV" label $LABEL >/dev/null 2>&1
	log info "$SOFTUSB configured to use device labeled $LABEL"
}

softusb_setreset() {
	case $1 in
	usb)	"$SETENV" busreset "usb reset" >/dev/null 2>&1;;
	ide)	"$SETENV" busreset "ide reset" >/dev/null 2>&1;;
	both)	"$SETENV" busreset "ide reset; usb reset" >/dev/null 2>&1;;
	*)		log error "Unknown option $1"; exit 1;;
	esac

	log info "$SOFTUSB configured to reset $1 bus"
}

softusb_setbootmenu() {
	case $1 in
	[1-9]);;
	*) log error "Bad argument '$1'"; exit 1;;
	esac
	case $2 in
	[1-9]|[1-4][0-9]|50);;
	*) log error "Bad argument '$2'"; exit 1;;
	esac

	softusb_configinit
	softusb_configset "bootmenucmd" "bootmenu $1 $2"
	softusb_configcommit
	log info "SoftUSB2 bootmenu set to $1 positions with timeout $2 seconds"
}

softusb_setflash() {
	local ROOTFS="$2"
	[ -z "$ROOTFS" ] && local ROOTFS=/dev/mtdblock3
	if [ "$1" = "auto" ]; then
		log info "Detecting software type..."
		local TMPDIR="`\"$BUSYBOX\" mktemp -d -t`"
		DETECT=""
		if softusb_mount "$LABEL" "$TMPDIR"; then
			[ "$ROOTFS" = "/dev/mtdblock3" ] && [ -d "$TMPDIR/app" ] && "$BUSYBOX" mount -r /dev/mtdblock4 "$TMPDIR/app" 2>/dev/null
			softusb_detect "$TMPDIR"
		fi
		"$BUSYBOX" umount "$TMPDIR/app" 2>/dev/null
		softusb_umount
		if [ -z "$DETECT" ]; then
			log error "Could not determine software type"; exit 1
		fi
		log info "Found: $DETECT"
	else
		DETECT=$1
	fi
	
	case "$DETECT" in
	e2_64|neutrino_64)	IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=64m coprocessor_mem=4m@0x10000000,4m@0x10400000";;
	e2_128|neutrino_128)	IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=128m coprocessor_mem=4m@0x10000000,4m@0x10400000";;
	pvr104)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=40m bigphysarea=1280 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
	pvr105)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=38m bigphysarea=1280 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
	pvr106)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=36m bigphysarea=1280 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
	pvr107)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=35m bigphysarea=1280 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
	pvr200)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=41m bigphysarea=2600 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
	pvr202)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=41m bigphysarea=2984 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
	*)			log error "Unknown image type $DETECT"; exit 1;;
	esac
	IMAGE_BOOTARGS="$IMAGE_BOOTARGS ip=\$ipaddr:\$serverip:\$gateway:$netmask:box:eth0:off nwhwconf=device:eth0,hwaddr:\$ethaddr"

	"$SETENV" 'bootdesc_0' "Flash" >/dev/null 2>&1
	"$SETENV" 'bootargs_0' "setenv bootargs '$IMAGE_BOOTARGS'" >/dev/null 2>&1
	"$SETENV" 'bootcmd_0' "run bootargs_0; bootm A0040000" >/dev/null 2>&1

	log info "$SOFTUSB flash configured to start $DETECT software, from $ROOTFS"
}

softusb_setenv() {
	"$SETENV" $1 "$2" >/dev/null 2>&1
	log info "Uboot entry '$1' set to '$2'"
}

softusb_setdefault() {
	case $1 in
	[0-9]);;
	*) log error "Bad argument '$1'"; exit 1;;
	esac
	"$SETENV" bootdesc_default "$1" >/dev/null 2>&1
	log info "Default menu item set to: $1"
}

softusb_savelastboot() {
	local SAVEDEF
	case $1 in
	yes)	SAVEDEF=1;;
	no)	SAVEDEF=0;;
	*) log error "Bad argument '$1'"; exit 1;;
	esac
	"$SETENV" bootdesc_savelast "$SAVEDEF" >/dev/null 2>&1
	log info "Save last selection set to: $1"
}

softusb_setnet() {
	local IPADDR="$1"
	local NETMASK="$2"
	local GATEWAY="$3"
	local SERVER="$4"
	if [ -z "$IPADDR" ] || [ -z "$NETMASK" ]; then
		softusb_usage
	else
		[ -z "$SERVER" ] && SERVER="$3"
		"$SETENV" ipaddr "$IPADDR" >/dev/null 2>&1
		"$SETENV" netmask "$NETMASK" >/dev/null 2>&1
		"$SETENV" gateway "$GATEWAY" >/dev/null 2>&1
		"$SETENV" serverip "$SERVER" >/dev/null 2>&1
		sync && sleep 1
		log info "Uboot network configured\nIP address: $IPADDR\nNetmask: $NETMASK\nGateway: $GATEWAY\nServer: $SERVER"
	fi
}

softusb_setmac() {
	local MAC="$1"
	"$SETENV" ethaddr "$MAC" >/dev/null 2>&1
	log info "Uboot (${UBOOT}) network mac set to: $MAC"
}

softusb_mkimage() {
	local TARFILE="$1"
	local IMGTYPE="$2"
	local IMGPATH="$3"
	local IMGNAME="$4"
	local SIZE=128
	SIZE=$(($SIZE*16))
	#sys prepare
	[ -f /etc/mtab ] || ln -s /proc/mounts /etc/mtab

	if [ "$IMGTYPE" = "file" ]; then
		log info "Creating partition file..."
		dd if=/dev/zero bs=64k count=$SIZE 2>/dev/null | "$BUSYBOX" pipe_progress | dd of="$IMGPATH/$IMGNAME" bs=64k 2>/dev/null
		if [ $? != 0 ]; then
			log error "Error creating file"; return 1
		fi
		sync && sleep 1
		local LOOPDEV="`\"$BUSYBOX\" losetup -f`"
		if [ -z "$LOOPDEV" ]; then
			log error "No free loop devices"; return 1
		fi
		"$BUSYBOX" losetup "$LOOPDEV" "$IMGPATH/$IMGNAME"
		log info "Formating partition file..."
		if ! "$BUSYBOX" mke2fs -j -q "$LOOPDEV"; then
			"$BUSYBOX" losetup -d "$LOOPDEV"
			log error "Error formating file"; return 1
		fi
		sync && sleep 1
		log info "Mounting partition file..."
		local TMPDIR="`\"$BUSYBOX\" mktemp -d -t`"
		mkdir -p "$TMPDIR"
		if ! "$BUSYBOX" mount "$LOOPDEV" "$TMPDIR"; then
			"$BUSYBOX" losetup -d "$LOOPDEV"
			log error "Error mounting image"; return 1
		fi
	elif [ "$IMGTYPE" = "dir" ]; then
		local TMPDIR="$IMGPATH/$IMGNAME"
		mkdir -p "$TMPDIR"
	else
		log info "Mounting partition..."
		local TMPDIR="`\"$BUSYBOX\" mktemp -d -t`"
		mkdir -p "$TMPDIR"
		if ! "$BUSYBOX" mount "$IMGNAME" "$TMPDIR"; then
			log error "Error mounting partition"; return 1
		fi
	fi
	sync
	log info "Extracting image..."
	dd if="$TARFILE" bs=64k 2>/dev/null | "$BUSYBOX" pipe_progress | "$BUSYBOX" tar -xzf - -C "$TMPDIR" 2>/dev/null
	if [ $? != 0 ]; then
		if [ "$IMGTYPE" = "file" ]; then
			"$BUSYBOX" umount "$TMPDIR"
			"$BUSYBOX" losetup -d "$LOOPDEV"
		fi
		log error "Error extracting image"; return 1
	fi
	if [ -d "$TMPDIR/dev_org" ]; then
		mkdir -p "$TMPDIR/dev"
		mkdir -p "$TMPDIR/mnt"
		mkdir -p "$TMPDIR/proc"
		mkdir -p "$TMPDIR/ramdisk"
		mkdir -p "$TMPDIR/var"
		mkdir -p "$TMPDIR/tmp"
	fi
	sync && sleep 1
	if [ ! "$IMGTYPE" = "part" ]; then
		if [ -f "$TMPDIR/boot/uImage" ]; then
			mv "$TMPDIR/boot/uImage" "$IMGPATH/uImage" >/dev/null 2>&1
		else
			log warning "Could not find uImage in archive"
		fi
	fi
	if [ "$IMGTYPE" = "file" ] || [ "$IMGTYPE" = "part" ]; then
		"$BUSYBOX" umount "$TMPDIR"
		[ "$IMGTYPE" = "file" ] && "$BUSYBOX" losetup -d "$LOOPDEV"
		rmdir "$TMPDIR"
	fi
	log info "Image $IMGNAME created successfully!"
}

softusb_mkswap() {
	local SIZE=$1
	if [ -z $SIZE ]; then
		SIZE=64
	fi
	if [ ! $SIZE -eq $SIZE 2>/dev/null ]; then
		log error "Bad argument $SIZE"; exit 1
	fi
	SIZE=$(($SIZE*16))

	if [ -f "$DIR/images/swap.part" ]; then
		log error "File swap.part already exists\nDelete it manually and retry"; exit 1
	fi

	log info "Creating swap file..."
	if ! dd if=/dev/zero of="$DIR/images/swap.part" bs=64k count=$SIZE 2>/dev/null; then
		log error "Error creating file"; exit 1
	fi
	sync
	log info "Formating swap file..."
	if ! "$BUSYBOX" mkswap "$DIR/images/swap.part"; then
		log error "Error formating file"; exit 1
	fi
	sync
	log info "Swap file created successfully!"
}

softusb_update() {
	local MENUNUMBERS="123456789"
	#First pass to collect done images and menu indexes

	for IMAGE_PATH in "$DIR/images"/*; do
		if [ -d "$IMAGE_PATH" ] && [ -f "$IMAGE_PATH/image.conf" ]; then
			unset IMAGE_NO
			. "$IMAGE_PATH/image.conf"
			[ ! -z "$IMAGE_NO" ] && MENUNUMBERS=`echo "$MENUNUMBERS" | sed "s/$IMAGE_NO//g" 2>/dev/null`
		fi
	done

	for IMAGE_PATH in "$DIR/images"/*; do
		if [ -d "$IMAGE_PATH" ]; then
			unset IMAGE_NO
			unset IMAGE_NAME
			unset IMAGE_BOOTARGS
			unset IMAGE_BOOTARGS_CUSTOM
			unset IMAGE_TYPE
			unset IMAGE_ROOTDELAY
			unset IMAGE_FASTBOOT
			unset IMAGE_INITRAM
			unset IMAGE_MTD_SWAP
			unset IMAGE_MTD_MINIFO
			unset IMAGE_PART
			unset IMAGE_UBOOTLOAD
			unset IMAGE_SOFT
			unset IMG_KER
			unset IMG_APP
			unset IMG_CONF
			unset IMG_DAT
			unset IMG_ROOT
			unset IMG_BTCF
			unset IMG_EME
			
			IMAGE_DIR="`\"$BUSYBOX\" basename \"$IMAGE_PATH\"`"
			IMAGE_CONF="$IMAGE_PATH/image.conf"
			[ -f "$IMAGE_CONF" ] && . "$IMAGE_CONF"
			log info "Processing $IMAGE_DIR..."
			#Determinate image type
			local ARCH_TYPE=""
			for o in "$IMAGE_PATH"/*; do
				if [ -z "$ARCH_TYPE" ]; then
					case "$o" in
					*.tar.gz)	IMAGE_TAR="$o"; ARCH_TYPE="tar";;
					*.tar.bz2)	IMAGE_TAR="$o"; ARCH_TYPE="tar";;
					*.tar.lzma)	IMAGE_TAR="$o"; ARCH_TYPE="tar";;
					esac
				fi
				if [ -z "$ARCH_TYPE" ] || [ "$ARCH_TYPE" = "img" ]; then
					case "$o" in
					*.img)
						local IMG_HEADER="`dd if=\"$o\" bs=4 skip=3 count=1 2>/dev/null`"
						case "$IMG_HEADER" in
							.ker)	local IMG_KER="$o";;
							.app)	local IMG_APP="$o";;
							conf)	local IMG_CONF="$o";;
							.dat)	local IMG_DAT="$o";;
							root)	local IMG_ROOT="$o";;
							btcf)	local IMG_BTCF="$o";;
							.eme)	local IMG_EME="$o";;
						esac
					;;
					esac
				fi
			done
			[ ! -z "$IMG_KER" ] && [ ! -z "$IMG_APP" ] && [ ! -z "$IMG_CONF" ] && [ ! -z "$IMG_DAT" ] && [ ! -z "$IMG_ROOT" ] && ARCH_TYPE="img"
			if [ ! -z "$ARCH_TYPE" ]; then
				case "$ARCH_TYPE" in
					tar)
						local IMAGE_TARGET=""
						if [ -z "$IMAGE_TYPE" ]; then
							local IMAGE_DEV="`\"$BUSYBOX\" mountpoint -n \"$IMAGE_PATH\" | cut -d\  -f1`"
							local IMAGE_FS=""
							for o in `"$BUSYBOX" blkid "$IMAGE_DEV"`; do
								case "$o" in
									TYPE=*)	IMAGE_FS=`echo ${o#TYPE=} | sed 's/"//g'`
								esac
							done

							case "$IMAGE_FS" in
								ext*)	IMAGE_TYPE=dir; IMAGE_TARGET="rootfs";;
								*)	IMAGE_TYPE=file; IMAGE_TARGET="rootfs.part";;
							esac
						fi

						case "$IMAGE_TYPE" in
							dir)	IMAGE_TARGET="rootfs";;
							file)	IMAGE_TARGET="rootfs.part";;
							part)	IMAGE_TARGET="$IMAGE_PART";;
							*)	log warning "Unknown image type: $IMAGE_TYPE";;
						esac

						if [ "$IMAGE_TYPE" = "part" ]; then
							if [ -f "$IMAGE_PATH/.extract" ]; then
								softusb_mkimage "$IMAGE_TAR" "$IMAGE_TYPE" "$IMAGE_PATH" "$IMAGE_TARGET" && rm -f "$IMAGE_PATH/.extract" || IMAGE_TYPE=""
							fi
						elif [ ! -z "$IMAGE_TARGET" ] && [ ! -e "$IMAGE_PATH/$IMAGE_TARGET" ]; then
							softusb_mkimage "$IMAGE_TAR" "$IMAGE_TYPE" "$IMAGE_PATH" "$IMAGE_TARGET" || IMAGE_TYPE=""
						fi
					;;
					img)
						IMAGE_TYPE=mtd
						[ -f "$IMAGE_PATH/uImage" ] || dd if="$IMG_KER" bs=16 skip=1 2>/dev/null | "$BUSYBOX" pipe_progress | dd of="$IMAGE_PATH/uImage" bs=64k 2>/dev/null
					;;
				esac
			fi

			if [ ! -z "$IMAGE_TYPE" ]; then
				#Save config
				[ -z "$IMAGE_NO" ] && IMAGE_NO="`echo $MENUNUMBERS | cut -c1`"				
				[ -z "$IMAGE_NAME" ] && IMAGE_NAME="$IMAGE_DIR"
				[ -z "$IMAGE_FASTBOOT" ] && IMAGE_FASTBOOT="No"
				[ -z "$IMAGE_INITRAM" ] && IMAGE_INITRAM="Yes"
				if [ -z "$IMAGE_ROOTDELAY" ]; then
					[ "$IMAGE_INITRAM" = "Yes" ] && IMAGE_ROOTDELAY=0 || IMAGE_ROOTDELAY=6
				fi
				[ -z "$IMAGE_MTD_SWAP" ] && IMAGE_MTD_SWAP="No"
				[ -z "$IMAGE_MTD_MINIFO" ] && IMAGE_MTD_MINIFO="No"
				if [ -z "$IMAGE_BOOTARGS" ]; then
					local TMPDIR="`\"$BUSYBOX\" mktemp -d -t`"
					unset DETECT
					case "$IMAGE_TYPE" in
						file)
							if [ -f "$IMAGE_PATH/rootfs.part" ]; then
								local LOOPDEV="`\"$BUSYBOX\" losetup -f`"
								if [ ! -z "$LOOPDEV" ]; then
									"$BUSYBOX" losetup "$LOOPDEV" "$IMAGE_PATH/rootfs.part"
									if softusb_mount "$LOOPDEV" "$TMPDIR"; then
										softusb_detect "$TMPDIR"
									fi
									softusb_umount "$LOOPDEV" "$TMPDIR"
									"$BUSYBOX" losetup -d "$LOOPDEV" 2>/dev/null
								fi
							fi
						;;
						dir)
							softusb_detect "$IMAGE_PATH/rootfs"
						;;
						part)
							if softusb_mount "$IMAGE_PART" "$TMPDIR"; then
								softusb_detect "$TMPDIR"
							fi
							softusb_umount "$IMAGE_PART" "$TMPDIR"
						;;
						mtd)
							if [ -f "$IMAGE_PATH/$IMG_APP" ]; then
								local LOOPDEV="`\"$BUSYBOX\" losetup -f`"
								if [ ! -z "$LOOPDEV" ]; then
									"$BUSYBOX" losetup -o 16 "$LOOPDEV" "$IMAGE_PATH/$IMG_APP"
									mkdir -p "$TMPDIR/app"
									if softusb_mount "$LOOPDEV" "$TMPDIR/app"; then
										softusb_detect "$TMPDIR"
									fi
									softusb_umount "$LOOPDEV" "$TMPDIR/app"
									rmdir "$TMPDIR" 2>/dev/null
									"$BUSYBOX" losetup -d "$LOOPDEV" 2>/dev/null
								fi
							fi
						;;
					esac
					if [ -z "$DETECT" ]; then
						if [ -z "$IMAGE_SOFT" ]; then
							DETECT="e2_$BOXMEM"
						else
							DETECT="$IMAGE_SOFT"
						fi
						log warning "Could not determine software type, assuming $DETECT"
					else
						case "$DETECT" in
						e2*)		log info "Found Enigma2";;
						pvr*)		log info "Found Kathrein software";;
						neutrino*)	log info "Found Neutrino";;
						vdr*)		log info "Found VDR";;
						esac
					fi
					
					if [ "$IMAGE_TYPE" = "part" ]; then
						argsroot="$IMAGE_PART"
					else
						argsroot="LABEL=\$label image=$IMAGE_DIR"
					fi
					
					case "$DETECT" in
					e2_64|neutrino_64)	IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=64m coprocessor_mem=4m@0x10000000,4m@0x10400000";;
					e2_128|neutrino_128)	IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=128m coprocessor_mem=4m@0x10000000,4m@0x10400000";;
					pvr104)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=40m bigphysarea=1280 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
					pvr105)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=38m bigphysarea=1280 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
					pvr106)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=36m bigphysarea=1280 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
					pvr107)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=35m bigphysarea=1280 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
					pvr200)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=41m bigphysarea=2600 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
					pvr202)			IMAGE_BOOTARGS="console=ttyAS0,115200 root=$argsroot rw mem=41m bigphysarea=2984 coprocessor_mem=2m@0x04000000,2m@0x04200000";;
					esac
					IMAGE_BOOTARGS="$IMAGE_BOOTARGS ip=\$ipaddr:\$serverip:\$gateway:$netmask:box:eth0:off nwhwconf=device:eth0,hwaddr:\$ethaddr"
					[ $IMAGE_ROOTDELAY -gt 0 ] && IMAGE_BOOTARGS="$IMAGE_BOOTARGS rootdelay=$IMAGE_ROOTDELAY"
					[ "$IMAGE_TYPE" = 'mtd' ] && IMAGE_BOOTARGS="$IMAGE_BOOTARGS mtdparts=Onboard_Flash:128k(Boot_firmware:0xA000.0000-0xA001.FFFF)ro,-(Flash_wo/_Bootloader:0xA002.0000-0xA0FF.FFFF)"
				fi
				echo "IMAGE_NO=$IMAGE_NO" > "$IMAGE_CONF"
				echo "IMAGE_NAME=$IMAGE_NAME" >> "$IMAGE_CONF"
				echo "IMAGE_BOOTARGS='$IMAGE_BOOTARGS'" >> "$IMAGE_CONF"
				echo "IMAGE_TYPE=$IMAGE_TYPE" >> "$IMAGE_CONF"
				echo "IMAGE_ROOTDELAY=$IMAGE_ROOTDELAY" >> "$IMAGE_CONF"
				echo "IMAGE_FASTBOOT=$IMAGE_FASTBOOT" >> "$IMAGE_CONF"
				if [ "$IMAGE_TYPE" != "part" ] && [ "$IMAGE_INITRAM" != Yes ]; then
					log warning "$IMAGE_TYPE requires InitRAM"
					IMAGE_INITRAM=Yes
				fi
				echo "IMAGE_INITRAM=$IMAGE_INITRAM" >> "$IMAGE_CONF"
				echo "IMAGE_MTD_SWAP=$IMAGE_MTD_SWAP" >> "$IMAGE_CONF"
				echo "IMAGE_MTD_MINIFO=$IMAGE_MTD_MINIFO" >> "$IMAGE_CONF"
				[ -z "$IMAGE_PART" ] || echo "IMAGE_PART=$IMAGE_PART" >> "$IMAGE_CONF"
				[ -z "$IMAGE_UBOOTLOAD" ] || echo "IMAGE_UBOOTLOAD='$IMAGE_UBOOTLOAD'" >> "$IMAGE_CONF"
				[ -z "$IMAGE_SOFT" ] || echo "IMAGE_SOFT=$IMAGE_SOFT" >> "$IMAGE_CONF"

				MENUNUMBERS=`echo $MENUNUMBERS | sed "s/$IMAGE_NO//g" 2>/dev/null`
			fi
		fi
	done

	log info "Updating configuration..."
	rm -f "$DIR/config.ub" 2>/dev/null
	softusb_configinit
	#Remove unused items
	while [ ! -z "$MENUNUMBERS" ]; do
		tmpNO="`echo $MENUNUMBERS | cut -c1`"
		softusb_configsetclear "bootcmd_$tmpNO"
		softusb_configsetclear "bootdesc_$tmpNO"
		softusb_configsetclear "bootargs_$tmpNO"
		MENUNUMBERS=`echo $MENUNUMBERS | sed "s/$tmpNO//g" 2>/dev/null`
	done

	for IMAGE_PATH in "$DIR/images"/*; do
		#Tricky way to get absolute path by device
		[ "`echo \"$IMAGE_PATH\" | cut -c-2`" = "./" ] && IMAGE_PATH="`pwd``echo \"$IMAGE_PATH\" | cut -c2-`"
		IMAGE_MP_DEV="`\"$BUSYBOX\" mountpoint -n \"$IMAGE_PATH\" | cut -f1 -d\ `"
		IMAGE_MP="`grep \"$IMAGE_MP_DEV\" /proc/mounts | cut -f2 -d\ `"		
		rm -f /tmp/.image_mpe >/dev/null 2>&1
		echo "$IMAGE_MP" | while read line; do
			IMAGE_MPE="`echo -e \"$line\"`"
			if [ "$IMAGE_MPE" != '/' ] && echo "$IMAGE_PATH" | grep -q "$IMAGE_MPE"; then
				echo "$IMAGE_MPE" > /tmp/.image_mpe
			fi
		done
		IMAGE_MP="`cat /tmp/.image_mpe`"
		IMAGE_RELPATH="`echo \"$IMAGE_PATH\" | sed -e \"s#^$IMAGE_MP##\"`"

		if [ -d "$IMAGE_PATH" ]; then
			IMAGE_DIR="`\"$BUSYBOX\" basename \"$IMAGE_PATH\"`"
			if [ -f "$IMAGE_PATH/image.conf" ]; then
				unset IMAGE_NO
				unset IMAGE_NAME
				unset IMAGE_BOOTARGS
				unset IMAGE_TYPE
				unset IMAGE_ROOTDELAY
				unset IMAGE_FASTBOOT
				unset IMAGE_INITRAM
				unset IMAGE_MTD_SWAP
				unset IMAGE_MTD_MINIFO
				unset IMAGE_PART
				unset IMAGE_UBOOTLOAD
				unset IMAGE_SOFT

				. "$IMAGE_PATH/image.conf"

				log info "Processing '$IMAGE_DIR' -> '$IMAGE_NAME'"
				softusb_configset "bootdesc_$IMAGE_NO" "$IMAGE_NAME"
				if [ "$IMAGE_TYPE" = "part" ]; then
					ubootload="$IMAGE_UBOOTLOAD a5000000 /boot/uImage"
				else
					ubootload="\$fsload \$fsinterface \$fsdev a5000000 $IMAGE_RELPATH/uImage"
				fi
				softusb_configset "bootargs_${IMAGE_NO}" "setenv bootargs $IMAGE_BOOTARGS"
				if [ "$IMAGE_FASTBOOT" = "Yes" ]; then
					ubootaddr='a0040000'
					ubootload=''
				else
					ubootaddr='a5000000'
				fi
				if [ "$IMAGE_INITRAM" = "Yes" ]; then
					ubootboot="$ubootload; run softusb; bootm $ubootaddr a5300000"
				else
					ubootboot="$ubootload; bootm $ubootaddr"
				fi
				softusb_configset "bootcmd_$IMAGE_NO" "run bootargs_$IMAGE_NO; $ubootboot"
			else
				log info "Skipping '$IMAGE_DIR'"
			fi
		fi
	done
	softusb_configcommit
}

softusb_checkuboot(){
	UBOOT1_MTD=mtd0
	UBOOT1_MTDBLOCK=mtdblock0
	UBOOT1CONF_MTD=`grep "0xA002.0000-0xA003.FFFF" /proc/mtd 2>/dev/null | cut -d: -f1`
	UBOOT1CONF_MTDBLOCK="mtdblock`echo \"$UBOOT1CONF_MTD\" | cut -b4-`"
	UBOOT2_MTD=`grep "0xA0FD.0000-0xA0FF.FFFF" /proc/mtd 2>/dev/null | cut -d: -f1`
	UBOOT2_MTDBLOCK="mtdblock`echo \"$UBOOT2_MTD\" | cut -b4-`"
	UBOOT2CONF_MTD=`grep "0xA0FC.0000-0xA0FC.FFFF" /proc/mtd 2>/dev/null | cut -d: -f1`
	UBOOT2CONF_MTDBLOCK="mtdblock`echo \"$UBOOT2CONF_MTD\" | cut -b4-`"
	if [ "$1" = "2" ] && [ ! -z $UBOOT2_MTD ] && [ ! -z $UBOOT2CONF_MTD ] && [ -b /dev/$UBOOT2_MTDBLOCK ] && [ -b /dev/$UBOOT2CONF_MTDBLOCK ]; then
		return 0
	elif [ ! -z $UBOOT1CONF_MTD ] && [ -b /dev/$UBOOT1CONF_MTDBLOCK ]; then
		return 0
	fi
	return 1
}

softusb_envuboot(){
	if [ $CHECKUBOOT -eq 0 ]; then
		echo "/dev/$UBOOT1CONF_MTD 0x00000000 0x00010000 0x10000" > /tmp/su_env.config
	fi
}

softusb_status(){
if [ $CHECKUBOOT -eq 0 ]; then
#network
	local ETHADDR="`\"$PRINTENV\" ethaddr 2>/dev/null | cut -d= -f2-`"
	local IPADDR="`\"$PRINTENV\" ipaddr 2>/dev/null | cut -d= -f2-`"
	local NETMASK="`\"$PRINTENV\" netmask 2>/dev/null | cut -d= -f2-`"
	local GATEWAY="`\"$PRINTENV\" gateway 2>/dev/null | cut -d= -f2-`"
	local SERVER="`\"$PRINTENV\" serverip 2>/dev/null | cut -d= -f2-`"
	log info "--==== NETWORK ==--
	MAC:\t$ETHADDR
	IP:\t$IPADDR
	MASK:\t$NETMASK
	GATEWAY:\t$GATEWAY
	SERVER:\t$SERVER"
#flash
	local CUSTOMBOOT="`\"$PRINTENV\" customboot 2>/dev/null | cut -d= -f2-`"
	local BOOTARGS0="`\"$PRINTENV\" bootargs0 2>/dev/null | cut -d= -f2-`"
	if [ ! -z "$CUSTOMBOOT" ]; then
		log info "--== FLASH CONFIG ==--
	BOOTCMD:\t$CUSTOMBOOT
	BOOTARGS:\t$BOOTARGS0"
	fi
fi
if [ -f "$DIR/config.ub" ]; then
	softusb_configinit
	#bootmenu
	local BOOTMENU="`cat /tmp/config.tmp | grep \"setenv bootmenucmd\" | cut -d\' -f2-`"
	if [ -z "$BOOTMENU" ]; then
		log info "--== BOOTMENU ==--
	Bootmenu not configured"
	else
		local NUM="`echo \"$BOOTMENU\" | cut -d\  -f 2`"
		local TIMEOUT="`echo \"$BOOTMENU\" | cut -d\  -f 3`"
		log info "--== BOOTMENU ==--
	NUMBER:\t$NUM
	TIMEOUT:\t$TIMEOUT"
	fi
	#image
	log info "--== IMAGES ==--"
	for i in 1 2 3 4 5 6 7 8 9; do
		local IMAGENAME="`cat /tmp/config.tmp | grep \"setenv customname$i\" | cut -c20-`"
		if [ ! -z "$IMAGENAME" ]; then
			log "\tIMAGE$i:\t$IMAGENAME"
		fi
	done
fi
}

#load config
DIR="`dirname \"$0\" 2>/dev/null`"
if [ -z "$DIR" ]; then
	DIR="${0%/*}"
fi
DIR="`cd \"$DIR\" && pwd`"
[ -f "$DIR/softusb.conf" ] && . "$DIR/softusb.conf"

BUSYBOX="$DIR/files/busybox"

[ -z "$BOXMEM" ] && BOXMEM=64
case "$BOXMEM" in
64|128);;
*)	log warning "Bad BOXMEM setting"; exit 1;;
esac

SETENV="$DIR/files/su_setenv"
PRINTENV="$DIR/files/su_printenv"

KERNEL_VER="`\"$BUSYBOX\" uname -r 2>/dev/null`"
if [ "$KERNEL_VER" = "2.6.17.14_stm22_0037" ]; then
	DEVNAME="`\"$BUSYBOX\" mountpoint -n \"$DIR\" | cut -d\  -f1`"
	"$BUSYBOX" mount -o remount,sync $DEVNAME 2>/dev/null
fi
! grep -q "loop" /proc/modules 2>/dev/null && [ -f "$DIR/files/modules/$KERNEL_VER/loop.ko" ] && "$BUSYBOX" insmod "$DIR/files/modules/$KERNEL_VER/loop.ko" 2>/dev/null
! grep -q "ufs910mtd" /proc/modules 2>/dev/null && [ -f "$DIR/files/modules/$KERNEL_VER/ufs910mtd.ko" ] && "$BUSYBOX" insmod "$DIR/files/modules/$KERNEL_VER/ufs910mtd.ko" secondstage=1 2>/dev/null && sleep 1

softusb_checkuboot
CHECKUBOOT=$?
softusb_envuboot

[ -d "$DIR/images" ] || mkdir -p "$DIR/images"

#commands
case $1 in
	install)	softusb_install "$2";;
	set)		softusb_set "$2" "$3" "$4" "$5" "$6" "$7" "$8" "$9";;
	update)		softusb_update;;
	uninstall)	softusb_uninstall;;
	status)		softusb_status;;
	*)		softusb_usage;;
esac
